# Implementation of LoCA

This repository is a Pytorch implementation of LoCA: Location-Aware Cosine Adaptation for Parameter-Efficient Fine-Tuning.

    
## Directory Description

The ``experiments`` folder contains the fine-tuning code for the main experiments in the paper, along with their execution scripts. These are used to reproduce the results shown in Tables 1, 2, and 3. All hyperparameters have been included in the execution scripts.

The ``peft`` folder is a modified version of the HuggingFace 🤗 PEFT library. We implement our LoCA within this framework. The specific modifications are as follows:

- ``peft/src/tuner/loca/layers.py``: This file contains the implementation of a LoCA Linear layer class, namely LoCALinear.

- ``peft/src/tuner/loca/config.py``: This file contains the configuration class corresponding to LoCA.

- ``peft/src/tuner/loca/model.py``: This file is a PeftModel class based on LoCALinear, which will be returned by the ``get_peft_model`` method.

- ``peft/src/tuner/loca/dct_utils.py``: This file contains the three different DCT implementations mentioned in the paper. One can also freely add any other more efficient DCT implementations here.

- ``peft/src/tuner/loca/bnb.py``: This file is the implementation of LoCALinear for int8 and int4 quantization scenarios.

- ``peft/src/peft/utils/save_and_load.py``: Methods for saving and loading the LoCA adapter have been added to this file.

## Environment Setup

To set up the Python environment for this project, we recommend using Conda. Follow these steps to create and activate the environment:

1. Create a new Conda environment with Python 3.9:

``` python
conda create -n loca python=3.9
```

2. Activate the newly created environment:

```python
source activate loca
```

3. Install dependencies for all experiments:

```python
cd ./peft
pip install -e .
pip install datasets scipy scikit-learn evaluate pillow torchvision sentencepiece bitsandbytes
```

## How to Run the Code

### GLUE Benchmark

To reproduce the results in Table 1. Please kindly move to ``experiments/GLUE/``.

``run_glue.py`` and ``run_glue_no_trainer.py`` are two fine-tuning pipelines with/without using ``transformers.Trainer``.

For faster fine-tuning, we recommend using ``run_glue_no_trainer.py``. We have provided all the fine-tuning scripts. Simply execute 

```python
./run_{task_name}_{base/large}.sh 
```
to run the code.

We have provided execution logs for all tasks in the ``logs_glue`` directory, which correspond to the results in Table 1.

### Instruction Tuning

To reproduce the results in Table 2. Please move to ``experiments/Instruction_Tuning`` and run a specific script to perform fine-tuning. For example,

```python
./llama-2-7b.sh
```

The fine-tuned adapter will be saved to ``output/llama2-7b/LoCA``.

To evaluate the model on mt_bench or vicuna_bench, please refer to FastChat (https://github.com/lm-sys/FastChat/tree/main/fastchat/llm_judge) for details.

### Image Classification

Similarly, please move to ``Image_Classification`` and run the prepared scripts to reproduce the results in Table 3. Here we provide three examples, i.e., DTD, StanfordCars and FGVC, which are the most challenging datasets in our evaluation. For example,

```python
./run_stanfordcars.sh
```

The running logs and results are also provided in ``results``. One can test any other fine-tuning methods or datasets by changing the ``mode`` or the ``dataset-name`` argument in the script.

### Other Fine-tuning Tasks

Based on the PEFT library, our LoCA can be easily applied to various fine-tuning tasks by adding just **two extra lines of code** to the original training script. For example, 


```python
from peft import LoCAConfig, get_peft_model
# Original code to load the model
model = AutoModelForSequenceClassification.from_pretrained("xxx")

# Add these two lines to apply LoCA
config = LoCAConfig(
            target_modules=["query", "value"],
            modules_to_save=["classifier"],
            n_frequency=args.n_frequency,
            loca_dropout = args_out.loca_dropout,
            learn_location_iter = args_out.learn_location_iter,
            dct_mode = args_out.loca_dct_mode,
            scale=args_out.scale
            )
model = get_peft_model(model, config)
# Continue with the rest of your training code
```
**Tips:** For instruction-tuning a Stable Diffusion, one can simply refer to [https://huggingface.co/blog/instruction-tuning-sd] to prepare the datasets and training script, and apply the above code to fine-tune a user-specific model. We have shown the results in Appendix K.

## Acknowledgments
This project builds upon the invaluable contributions of following open-source projects:

1. PEFT (https://github.com/huggingface/peft)
2. FourierFT (https://github.com/Chaos96/fourierft)
3. Stanford Alpaca (https://github.com/tatsu-lab/stanford_alpaca)

We express our sincere gratitude to the talented authors who have generously shared their source code with the public, enabling us to leverage their work in our own endeavor.